package lambda;

import java.util.*;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * @author Georgiy Korneev (kgeorgiy@kgeorgiy.info)
 */
public enum Implementation {
    ;
    
    public static void main(final String[] args) {
        characteristics();
        spliterators();
        collectors();
    }

    private static void characteristics() {
        System.out.println("--- characteristics");
        dumpCharacteristics("List.of()", List.of());
        dumpCharacteristics("List.of(1, 2)", List.of(1, 2));
        dumpCharacteristics("Arrays.asList(1, 2)", Arrays.asList(1, 2));
        dumpCharacteristics("Set.of(1, 2)", Set.of(1, 2));
        dumpCharacteristics("new ArrayList<>(List.of(1, 2))", new ArrayList<>(List.of(1, 2)));
        dumpCharacteristics("lines()", "a\nb".lines().spliterator());

        System.out.println("--- binding");
        dumpBinding("ArrayList", ArrayList::new);
        dumpBinding("CopyOnWriteArrayList", CopyOnWriteArrayList::new);
        dumpBinding("ConcurrentSkipListSet", ConcurrentSkipListSet::new);
    }

    private static void dumpBinding(final String name, final Function<List<Integer>, Collection<Integer>> cons) {
        System.out.format("\t%s%n", name);
        final Collection<Integer> values = cons.apply(List.of(1, 2));
        final Stream<Integer> stream = values.stream();
        values.add(3);
        System.out.format("\t\tbefore: %s%n", stream.collect(Collectors.toList()));

        System.out.format("\t\tconcurrent: %s%n", concurrentModification(cons.apply(List.of(1, 2))));
    }

    private static String concurrentModification(final Collection<Integer> coll) {
        try {
            return coll.stream().peek(coll::add).toList().toString();
        } catch (final ConcurrentModificationException e) {
            return "concurrent modification";
        }
    }

    private static void dumpCharacteristics(final String name, final Collection<?> values) {
        dumpCharacteristics(name, values.stream().spliterator());
    }

    private static void dumpCharacteristics(final String name, final Spliterator<?> sit) {
        System.out.format("\t%s: %s%n", name, Arrays.stream(Characteristic.values())
                .filter(c -> sit.hasCharacteristics(c.value))
                .collect(Collectors.toList()));
    }

    enum Characteristic {
        DISTINCT(Spliterator.DISTINCT),
        IMMUTABLE(Spliterator.IMMUTABLE),
        SORTED(Spliterator.SORTED),
        ORDERED(Spliterator.ORDERED),
        SIZED(Spliterator.SIZED),
        SUBSIZED(Spliterator.SUBSIZED),
        NONNULL(Spliterator.NONNULL),
        CONCURRENT(Spliterator.CONCURRENT);

        private final int value;

        Characteristic(final int value) {
            this.value = value;
        }
    }

    private static void spliterators() {
        System.out.println("--- Chunked");
        final List<String> strings = List.of("a", "b", "ab", "aba", "abac", "abacaba", "cabacabad", "cabacabadaba");
        for (int i = 0; i < strings.size(); i++) {
            final List<String> values = strings.subList(0, i);
            System.out.format("\t%s -> %s%n", values, chunked(values.stream(), 3).collect(Collectors.toList()));
        }
    }

    private static void collectors() {
        final List<Student> students = List.of(new Student("a", "z"), new Student("b", "x"), new Student("c", "y"));

        System.out.println("--- Min names");
        System.out.println(students.stream().collect(MIN_NAMES));
        System.out.println(students.stream().collect(Implementation.tee(
                Collectors.mapping(Student::firstName, Collectors.minBy(Comparator.naturalOrder())),
                Collectors.mapping(Student::lastName, Collectors.minBy(Comparator.naturalOrder()))
        )));
        System.out.println(students.stream().collect(Collectors.teeing(
                Collectors.mapping(Student::firstName, Collectors.minBy(Comparator.naturalOrder())),
                Collectors.mapping(Student::lastName, Collectors.minBy(Comparator.naturalOrder())),
                (Optional<String> first, Optional<String> last) -> first.flatMap(f -> last.map(l -> new Student(f, l)))
        )));

        System.out.println("--- Has");
        System.out.println(students.stream().collect(has(student -> "a".equals(student.firstName()))));
        System.out.println(students.stream().collect(has(student -> "p".equals(student.firstName()))));
    }

    private static String min(final String a, final String b) {
        return a != null && a.compareTo(b) < 0 ? a : b;
    }

    private static final Collector<Student, String[], Student> MIN_NAMES = Collector.of(
            () -> new String[2],
            (mins, student) -> {
                mins[0] = min(mins[0], student.firstName());
                mins[1] = min(mins[1], student.lastName());
            },
            (mins1, mins2) -> {
                mins1[0] = min(mins1[0], mins2[0]);
                mins1[1] = min(mins1[1], mins2[1]);
                return mins1;
            },
            mins -> new Student(mins[0], mins[1])
    );

    private record Student(String firstName, String lastName) {
    }

    private static <T> Stream<List<T>> chunked(final Stream<? extends T> stream, final int n) {
        final Spliterator<? extends T> spliterator = stream.spliterator();
        return StreamSupport.stream(
                new Spliterators.AbstractSpliterator<>(
                        (spliterator.estimateSize() + n - 1) / n,
                        spliterator.characteristics() & ~Spliterator.SORTED
                ) {
                    @Override
                    public boolean tryAdvance(final Consumer<? super List<T>> action) {
                        final List<T> chunk = new ArrayList<>();
                        while (chunk.size() < n && spliterator.tryAdvance(chunk::add)) {
                        }
                        if (!chunk.isEmpty()) {
                            action.accept(chunk);
                        }
                        return !chunk.isEmpty();
                    }
                }, false
        );
    }

    private static <T> Collector<T, ?, Boolean> has(final Predicate<? super T> p) {
        class State {
            boolean found;
        }
        return Collector.of(
                State::new,
                (state, e) -> state.found |= p.test(e),
                (state1, state2) -> {
                    state1.found |= state2.found;
                    return state1;
                },
                state -> state.found
        );
    }


    public record Pair<F, S>(F first, S second) {}

    private static <T, F, S, FA, SA> Collector<T, Pair<FA, SA>, Pair<F, S>> tee(
            final Collector<T, FA, F> first,
            final Collector<T, SA, S> second
    ) {
        return Collector.of(
                () -> new Pair<>(first.supplier().get(), second.supplier().get()),
                (state, value) -> {
                    first.accumulator().accept(state.first, value);
                    second.accumulator().accept(state.second, value);
                },
                (state1, state2) -> new Pair<>(
                        first.combiner().apply(state1.first, state2.first),
                        second.combiner().apply(state1.second, state2.second)
                ),
                state -> new Pair<>(
                        first.finisher().apply(state.first),
                        second.finisher().apply(state.second)
                )
        );
    }
}
