package lambda;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Georgiy Korneev (kgeorgiy@kgeorgiy.info)
 */
public enum Words {
    ;

    public static void main(final String... args) throws IOException {
        final String input = log("Input string", "Monday's child is fair of face. Tuesday's child is classes of grace.");
        final List<String> words = log("Words", words(input).collect(Collectors.toList()));
        log("Containing 'e'", containing(words, "e"));
        log("Statistics", statistics(words));
        log("Ordered statistics", orderedStatistics(words));
        log("Uppercase statistics", uppercaseStatistics(words));
        log("Tree statistics", treeStatistics(Paths.get("."), ".java"));
    }

    private static List<String> containing(final List<String> words, final String substring) {
        return words.stream().filter(s -> s.contains(substring)).collect(Collectors.toList());
    }

    private static Stream<String> words(final String input) {
        return Arrays.stream(input.split("[^a-zA-Z']+"));
    }

    private static Map<String, Long> statistics(final List<String> words) {
        return words.stream().collect(Collectors.groupingBy(
                Function.identity(),
                Collectors.counting()
        ));
    }

    private static TreeMap<String, Long> orderedStatistics(final List<String> words) {
        return words.stream().collect(Collectors.groupingBy(
                Function.identity(),
                TreeMap::new,
                Collectors.counting()
        ));
    }

    private static Map<String, Long> uppercaseStatistics(final List<String> words) {
        return words.stream().collect(Collectors.groupingBy(
                String::toUpperCase,
                Collectors.counting()
        ));
    }

    private static Map<Path, Map<String, Long>> treeStatistics(final Path path, final String extension) {
        return IOSupplier.unchecked(() -> {
            try (final Stream<Path> files = Files.find(path, Integer.MAX_VALUE, (file, attrs) -> file.toString().endsWith(extension))) {
                return files
                        .collect(Collectors.toMap(
                                Function.identity(),
                                Words::fileStatistics
                        ));
            }
        });
    }

    private static Map<String, Long> fileStatistics(final Path path) {
        return IOSupplier.unchecked(() -> {
            try (final Stream<String> lines = Files.lines(path)) {
                return lines
                        .flatMap(Words::words)
                        .collect(Collectors.groupingBy(
                                Function.identity(),
                                Collectors.counting()
                        ));
            }
        });
    }

    private static <T> T log(final String name, final T value) {
        System.out.format("%20s: %s%n", name, value);
        return value;
    }
}
