package info.kgeorgiy.java.testModule;

import info.kgeorgiy.java.helloModule.service.HelloService;

import java.util.Arrays;
import java.util.ServiceLoader;
import java.util.function.Consumer;

public class Main {
    public static final String PREFIX = "info.kgeorgiy.java.helloModule.";

    public static final Consumer<Class<?>> CREATE_INSTANCE = token -> {
        try {
            token.getDeclaredConstructor().newInstance();
            System.out.format("    + Instance creation OK%n");
        } catch (final IllegalAccessException e) {
            System.out.format("    ? Default constructor access denied%n");
        } catch (final Exception e) {
            System.out.format("    ? Default constructor lookup failed: %s %s%n", e.getClass().getSimpleName(), e.getMessage());
        }
    };

    public static final Consumer<Class<?>> DUMP_FIELDS = token -> {

        for (final String mod : Arrays.asList("public", "protected", "packagePrivate", "private")) {
            final String field = mod + "Field";
            try {
                if (token.getDeclaredField(field).trySetAccessible()) {
                    System.out.format("    + Field %s access OK%n", field);
                } else {
                    System.out.format("    ? Field %s setAccessible failed%n", field);
                }
            } catch (final NoSuchFieldException e) {
                System.out.format("    - Field %s not found%n", field);
            }
        }
    };

    public static void main(final String... args) {
        System.out.println("Hello from " + Main.class.getSimpleName());

        exports();
        transitive();
        reflection();
        services();
    }

    private static void exports() {
        part("Class access");

        // Visible at compile time and allows full reflective access
        access(info.kgeorgiy.java.helloModule.exportedAndOpened.Public.class.getName());

        // Visible at compile time only
        access(info.kgeorgiy.java.helloModule.exported.Public.class.getName());

        // Allows only reflective access
        //access(info.kgeorgiy.java.helloModule.opened.Public.class.getName();
        access("info.kgeorgiy.java.helloModule.opened.Public");

        // Not visible and do not allows reflective access
        access("info.kgeorgiy.java.helloModule.nonOpened.Public");
    }

    private static void transitive() {
        part("Transitive requires");

        // Visible at compile time
        access(java.rmi.Remote.class.getName(), "java.rmi/Remote");

        // Not visible at compile time
        access("com.sun.net.httpserver.HttpServer", "jdk.httpserver/HttpServer");
    }

    private static void reflection() {
        part("Reflective class members access");
        for (final String pack : Arrays.asList("exportedAndOpened", "exported", "opened", "nonExportedOrOpened", "test")) {
            for (final String cls : Arrays.asList("Public", "PackagePrivate")) {
                access(PREFIX + pack + "." + cls, CREATE_INSTANCE, DUMP_FIELDS);
            }
        }
    }

    private static void part(final String name) {
        System.out.format("%n=== %s%n", name);
    }

    @SafeVarargs
    private static void access(final String className, final Consumer<Class<?>>... actions) {
        access(className, className.substring(PREFIX.length()), actions);
    }

    @SafeVarargs
    private static void access(final String className, final String name, final Consumer<Class<?>>... actions) {
        try {
            final Class<?> token = Class.forName(className);
            System.out.format("+ Class %s found%n", name);
            for (final Consumer<Class<?>> action : actions) {
                action.accept(token);
            }
        } catch (final ClassNotFoundException e) {
            System.out.format("- Class %s not found%n", name);
        } catch (final Exception e) {
            System.out.format("Error accessing class %s: %s (%s)%n", name, e.getClass().getSimpleName(), e.getMessage());
        }
    }

    private static void services() {
        part("Services");
        final ServiceLoader<HelloService> loader = ServiceLoader.load(HelloService.class);
        for (final HelloService service : loader) {
            System.out.format("    Class %s%n", service.getClass().getName().substring(PREFIX.length()));
            service.sayHello("Main");
        }
    }
}
