package reflection;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.Path;
import java.util.List;

/**
 * @author Georgiy Korneev (kgeorgiy@kgeorgiy.info)
 */
public final class ClassLoaderExample {
    private static final Path JAR = Path.of("reflection-example.jar");
    private static final URL URL;

    static {
        try {
            URL = JAR.toUri().toURL();
        } catch (final MalformedURLException e) {
            throw new AssertionError(e);
        }
    }

    public static void main(final String... args) {
        final List<String> classes = List.of(
                "Good",
                "Bad",
                "Missing",
                "BadInit",
                "NoMethods",
                "Inaccessible",
                "Abstract",
                "MethodThrows"
        );
        for (final String name : classes) {
            process("Classes$" + name);
        }
    }

    private static void process(final String className) {
        System.out.println("Processing " + className);
        final ClassLoader cl = new URLClassLoader(new URL[]{URL});
        final Class<?> clazz;
        System.out.println("\tLoading class " + className);
        try {
            clazz = cl.loadClass(className);
        } catch (final ClassNotFoundException e) {
            System.out.println("\t\tCannot load class " + className + ": " + exception(e));
            return;
        }

        System.out.println("\tRunning " + className + ".main(\"hello\")");
        try {
            final Method m = clazz.getDeclaredMethod("main", String[].class);
            m.invoke(null, (Object) new String[]{"world"});
        } catch (final NoSuchMethodException e) {
            System.out.println("\t\tNo method main found in " + clazz.getCanonicalName());
        } catch (final IllegalAccessException e) {
            System.out.println("\t\tInaccessible method main in " + clazz.getCanonicalName());
        } catch (final InvocationTargetException e) {
            System.out.println("\t\tMethod main in " + clazz.getCanonicalName() + " thrown " + exception(e.getCause()));
        }

        System.out.println("\tRunning " + className + ".run()");
        try {
            final Constructor<?> constructor = clazz.getDeclaredConstructor();
            final Runnable instance = (Runnable) constructor.newInstance();
            try {
                instance.run();
            } catch (final RuntimeException e) {
                System.out.println("\t\trun() of " + clazz.getCanonicalName() + " thrown " + exception(e));
            }
        } catch (final NoSuchMethodException e) {
            System.out.println("\t\tNo default constructor in " + clazz.getCanonicalName());
        } catch (final InstantiationException e) {
            System.out.println("\t\tClass " + clazz.getCanonicalName() + " is abstract");
        } catch (final IllegalAccessException e) {
            System.out.println("\t\tInaccessible default constructor in " + clazz.getCanonicalName());
        } catch (final InvocationTargetException | ExceptionInInitializerError e) {
            System.out.println("\t\tDefault constructor of " + clazz.getCanonicalName() + " thrown " + exception(e.getCause()));
        } catch (final ClassCastException e) {
            System.out.println("\t\tClass " + clazz.getCanonicalName() + " does not implements Runnable");
        }
    }

    private static String exception(final Throwable e) {
        return e.getClass().getSimpleName() + ": " + e.getMessage();
    }
}
